#define STRICT
#define VC_EXTRALEAN
#include <atlbase.h>
#include <atlapp.h>
CAppModule _Module;
#include <atlwin.h>
#include <windows.h>
#include <shlobj.h>
#include <shlwapi.h>
#include <stdio.h>
#include <stdlib.h>
#include <atlctrls.h>
#include <atlframe.h>
#include <atlcrack.h>
#include <atldlgs.h>
#include "../inc/toupcam.h"
#include "resource.h"
#include <deque>
#include <InitGuid.h>
#include <wincodec.h>
#include <wmsdkidl.h>

#define MSG_CAMEVENT			(WM_APP + 1)

#define TDIBWIDTHBYTES(bits)   (((bits) + 31) / 32 * 4)

class CMainFrame;

static BOOL SaveBmp(const wchar_t* strFilename, const void* pData, const BITMAPINFOHEADER* pHeader)
{
	FILE* fp = _wfopen(strFilename, L"wb");
	if (fp)
	{
		BITMAPFILEHEADER fheader = { 0 };
		fheader.bfType = 0x4d42;
		fheader.bfSize = sizeof(BITMAPFILEHEADER) + sizeof(BITMAPINFOHEADER) + pHeader->biSizeImage;
		fheader.bfOffBits = (DWORD)(sizeof(BITMAPFILEHEADER) + sizeof(BITMAPINFOHEADER));
		fwrite(&fheader, sizeof(fheader), 1, fp);
		fwrite(pHeader, 1, sizeof(BITMAPINFOHEADER), fp);
		fwrite(pData, 1, pHeader->biSizeImage, fp);
		fclose(fp);
		return TRUE;
	}
	return FALSE;
}

static BOOL SaveByWIC(const wchar_t* strFilename, const void* pData, const BITMAPINFOHEADER* pHeader)
{
	GUID guidContainerFormat;
	if (PathMatchSpec(strFilename, L"*.jpg"))
		guidContainerFormat = GUID_ContainerFormatJpeg;
	else if (PathMatchSpec(strFilename, L"*.png"))
		guidContainerFormat = GUID_ContainerFormatPng;
	else
		return FALSE;

	CComPtr<IWICImagingFactory> spIWICImagingFactory;
	HRESULT hr = CoCreateInstance(CLSID_WICImagingFactory, NULL, CLSCTX_INPROC_SERVER, __uuidof(IWICImagingFactory), (LPVOID*)&spIWICImagingFactory);
	if (FAILED(hr))
		return FALSE;

	CComPtr<IWICBitmapEncoder> spIWICBitmapEncoder;
	hr = spIWICImagingFactory->CreateEncoder(guidContainerFormat, NULL, &spIWICBitmapEncoder);
	if (FAILED(hr))
		return FALSE;

	CComPtr<IWICStream> spIWICStream;
	spIWICImagingFactory->CreateStream(&spIWICStream);
	if (FAILED(hr))
		return FALSE;

	hr = spIWICStream->InitializeFromFilename(strFilename, GENERIC_WRITE);
	if (FAILED(hr))
		return FALSE;

	hr = spIWICBitmapEncoder->Initialize(spIWICStream, WICBitmapEncoderNoCache);
	if (FAILED(hr))
		return FALSE;

	CComPtr<IWICBitmapFrameEncode> spIWICBitmapFrameEncode;
	CComPtr<IPropertyBag2> spIPropertyBag2;
	hr = spIWICBitmapEncoder->CreateNewFrame(&spIWICBitmapFrameEncode, &spIPropertyBag2);
	if (FAILED(hr))
		return FALSE;

	if (GUID_ContainerFormatJpeg == guidContainerFormat)
	{
		PROPBAG2 option = { 0 };
		option.pstrName = L"ImageQuality"; /* jpg quality, you can change this settin */
		CComVariant varValue(0.75f);
		spIPropertyBag2->Write(1, &option, &varValue);
	}
	hr = spIWICBitmapFrameEncode->Initialize(spIPropertyBag2);
	if (FAILED(hr))
		return FALSE;

	hr = spIWICBitmapFrameEncode->SetSize(pHeader->biWidth, pHeader->biHeight);
	if (FAILED(hr))
		return FALSE;

	WICPixelFormatGUID formatGUID = GUID_WICPixelFormat24bppBGR;
	hr = spIWICBitmapFrameEncode->SetPixelFormat(&formatGUID);
	if (FAILED(hr))
		return FALSE;

	hr = spIWICBitmapFrameEncode->WritePixels(pHeader->biHeight, TDIBWIDTHBYTES(pHeader->biWidth * pHeader->biBitCount), TDIBWIDTHBYTES(pHeader->biWidth * pHeader->biBitCount) * pHeader->biHeight, (BYTE*)pData);
	if (FAILED(hr))
		return FALSE;

	hr = spIWICBitmapFrameEncode->Commit();
	if (FAILED(hr))
		return FALSE;
	hr = spIWICBitmapEncoder->Commit();
	if (FAILED(hr))
		return FALSE;
	
	return TRUE;
}

class CExposureTimeDlg : public CDialogImpl<CExposureTimeDlg>
{
	HToupCam*	m_phToupCam;
public:
	enum { IDD = IDD_EXPOSURETIME };

	CExposureTimeDlg(HToupCam* phToupCam)
	: m_phToupCam(phToupCam)
	{
	}

	BEGIN_MSG_MAP(CExposureTimeDlg)
		MESSAGE_HANDLER(WM_INITDIALOG, OnInitDialog)
		COMMAND_ID_HANDLER(IDOK, OnOK)
		COMMAND_ID_HANDLER(IDCANCEL, OnCancel)
	END_MSG_MAP()

	LRESULT OnInitDialog(UINT /*uMsg*/, WPARAM /*wParam*/, LPARAM /*lParam*/, BOOL& /*bHandled*/)
	{
		CenterWindow(GetParent());

		unsigned nMin = 0, nMax = 0, nDef = 0, nTime = 0;
		if (*m_phToupCam)
		{
			if (SUCCEEDED(Toupcam_get_ExpTimeRange(*m_phToupCam, &nMin, &nMax, &nDef)))
			{
				CTrackBarCtrl ctrl(GetDlgItem(IDC_SLIDER1));
				ctrl.SetRangeMin(nMin);
				ctrl.SetRangeMax(nMax);

				if (SUCCEEDED(Toupcam_get_ExpoTime(*m_phToupCam, &nTime)))
					ctrl.SetPos(nTime);
			}
		}
		
		return TRUE;
	}

	LRESULT OnOK(WORD /*wNotifyCode*/, WORD wID, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		CTrackBarCtrl ctrl(GetDlgItem(IDC_SLIDER1));
		if (*m_phToupCam)
			Toupcam_put_ExpoTime(*m_phToupCam, ctrl.GetPos());

		EndDialog(wID);
		return 0;
	}

	LRESULT OnCancel(WORD /*wNotifyCode*/, WORD wID, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		EndDialog(wID);
		return 0;
	}
};

class CMainView : public CWindowImpl<CMainView>
{
	CMainFrame*	m_pMainFrame;

public:

	static ATL::CWndClassInfo& GetWndClassInfo()
	{
		static ATL::CWndClassInfo wc =
		{
			{ sizeof(WNDCLASSEX), CS_HREDRAW | CS_VREDRAW, StartWindowProc,
			  0, 0, NULL, NULL, NULL, (HBRUSH)NULL_BRUSH, NULL, NULL, NULL },
			NULL, NULL, IDC_ARROW, TRUE, 0, _T("")
		};
		return wc;
	}

	CMainView(CMainFrame* pMainFrame)
	: m_pMainFrame(pMainFrame)
	{
	}

	BEGIN_MSG_MAP(CMainView)
		MESSAGE_HANDLER(WM_PAINT, OnWmPaint)
		MESSAGE_HANDLER(WM_ERASEBKGND, OnEraseBkgnd)
	END_MSG_MAP()

	LRESULT OnWmPaint(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled);
	LRESULT OnEraseBkgnd(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
	{
		return 1;
	}
};

class CWmvRecord
{
	const LONG				m_lFrameWidth, m_lFrameHeight;
	CComPtr<IWMWriter>		m_spIWMWriter;
public:
	CWmvRecord(LONG lFrameWidth, LONG lFrameHeight)
	: m_lFrameWidth(lFrameWidth), m_lFrameHeight(lFrameHeight)
	{
	}

	BOOL StartRecord(const wchar_t* strFilename, DWORD dwBitrate)
	{
		CComPtr<IWMProfileManager> spIWMProfileManager;
		HRESULT hr = WMCreateProfileManager(&spIWMProfileManager);
		if (FAILED(hr))
			return FALSE;

		CComPtr<IWMCodecInfo> spIWMCodecInfo;
		hr = spIWMProfileManager->QueryInterface(__uuidof(IWMCodecInfo), (void**)&spIWMCodecInfo);
		if (FAILED(hr))
			return FALSE;

		DWORD cCodecs = 0;
        hr = spIWMCodecInfo->GetCodecInfoCount(WMMEDIATYPE_Video, &cCodecs);
		if (FAILED(hr))
			return FALSE;

		CComPtr<IWMStreamConfig> spIWMStreamConfig;
        //
        // Search from the last codec because the last codec usually 
        // is the newest codec. 
        //
        for (int i = cCodecs - 1; i >= 0; i--)
        {
            DWORD cFormats;
            hr = spIWMCodecInfo->GetCodecFormatCount(WMMEDIATYPE_Video, i, &cFormats);
			if (FAILED(hr))
				break;

            for (DWORD j = 0; j < cFormats; j++)
			{
                hr = spIWMCodecInfo->GetCodecFormat(WMMEDIATYPE_Video, i, j, &spIWMStreamConfig);
				if (FAILED(hr))
					break;

				hr = ConfigureInput(spIWMStreamConfig, dwBitrate);
				if (SUCCEEDED(hr))
					break;
				spIWMStreamConfig = NULL;
			}
			if (spIWMStreamConfig)
				break;
		}
		if (spIWMStreamConfig == NULL)
			return FALSE;

		CComPtr<IWMProfile> spIWMProfile;
		hr = spIWMProfileManager->CreateEmptyProfile(WMT_VER_8_0, &spIWMProfile);
		if (FAILED(hr))
			return FALSE;

		{
			CComPtr<IWMStreamConfig> spIWMStreamConfig2;
			hr = spIWMProfile->CreateNewStream(WMMEDIATYPE_Video, &spIWMStreamConfig2);
			if (FAILED(hr))
				return FALSE;

			WORD wStreamNum = 1;
			hr = spIWMStreamConfig2->GetStreamNumber(&wStreamNum);
			if (FAILED(hr))
				return FALSE;
			
			hr = spIWMStreamConfig->SetStreamNumber(wStreamNum);
			if (FAILED(hr))
				return FALSE;
		}
		spIWMStreamConfig->SetBitrate(dwBitrate);

		hr = spIWMProfile->AddStream(spIWMStreamConfig);
		if (FAILED(hr))
			return FALSE;

		hr = WMCreateWriter(NULL, &m_spIWMWriter);
		if (FAILED(hr))
			return FALSE;

		hr = m_spIWMWriter->SetProfile(spIWMProfile);
		if (FAILED(hr))
			return FALSE;

		hr = SetInputProps();
		if (FAILED(hr))
			return FALSE;

		hr = m_spIWMWriter->SetOutputFilename(strFilename);
		if (FAILED(hr))
			return FALSE;

		hr = m_spIWMWriter->BeginWriting();
		if (FAILED(hr))
			return FALSE;

		{
			CComPtr<IWMWriterAdvanced> spIWMWriterAdvanced;
			m_spIWMWriter->QueryInterface(__uuidof(IWMWriterAdvanced), (void**)&spIWMWriterAdvanced);
			if (spIWMWriterAdvanced)
				spIWMWriterAdvanced->SetLiveSource(TRUE);
		}

		return TRUE;
	}

	void StopRecord()
	{
		if (m_spIWMWriter)
		{
			m_spIWMWriter->Flush();
			m_spIWMWriter->EndWriting();
			m_spIWMWriter = NULL; 
		}
	}

	BOOL WriteSample(const void* pData)
	{
		CComPtr<INSSBuffer> spINSSBuffer;
		if (SUCCEEDED(m_spIWMWriter->AllocateSample(TDIBWIDTHBYTES(m_lFrameWidth * 24) * m_lFrameHeight, &spINSSBuffer)))
		{
			BYTE* pBuffer = NULL;
			if (SUCCEEDED(spINSSBuffer->GetBuffer(&pBuffer)))
			{
				memcpy(pBuffer, pData, TDIBWIDTHBYTES(m_lFrameWidth * 24) * m_lFrameHeight);
				spINSSBuffer->SetLength(TDIBWIDTHBYTES(m_lFrameWidth * 24) * m_lFrameHeight);
				QWORD cnsSampleTime = GetTickCount();
				m_spIWMWriter->WriteSample(0, cnsSampleTime * 1000 * 10, 0, spINSSBuffer);
				return TRUE;
			}
		}

		return FALSE;
	}

private:
	HRESULT SetInputProps()
	{
		DWORD dwForamts = 0;
		HRESULT hr = m_spIWMWriter->GetInputFormatCount(0, &dwForamts);
		if (FAILED(hr))
			return hr;

		for (DWORD i = 0; i < dwForamts; ++i)
		{
			CComPtr<IWMInputMediaProps> spIWMInputMediaProps;
			hr = m_spIWMWriter->GetInputFormat(0, i, &spIWMInputMediaProps);
			if (FAILED(hr))
				return hr;

			DWORD cbSize = 0;
			hr = spIWMInputMediaProps->GetMediaType(NULL, &cbSize);
			if (FAILED(hr))
				return hr;

			WM_MEDIA_TYPE* pMediaType = (WM_MEDIA_TYPE*)alloca(cbSize);
			hr = spIWMInputMediaProps->GetMediaType(pMediaType, &cbSize);
			if (FAILED(hr))
				return hr;

			if (pMediaType->subtype == WMMEDIASUBTYPE_RGB24)
			{
				hr = spIWMInputMediaProps->SetMediaType(pMediaType);
				if (FAILED(hr))
					return hr;

				return m_spIWMWriter->SetInputProps(0, spIWMInputMediaProps);
			}
		}

		return E_FAIL;
	}

	HRESULT ConfigureInput(CComPtr<IWMStreamConfig>& spIWMStreamConfig, DWORD dwBitRate)
	{
		CComPtr<IWMVideoMediaProps> spIWMVideoMediaProps;
		HRESULT hr = spIWMStreamConfig->QueryInterface(__uuidof(IWMVideoMediaProps), (void**)&spIWMVideoMediaProps);
		if (FAILED(hr))
			return hr;

		DWORD cbMT = 0;
		hr = spIWMVideoMediaProps->GetMediaType(NULL, &cbMT);
		if (FAILED(hr))
			return hr;

		// Allocate memory for the media type structure.
		WM_MEDIA_TYPE* pType = (WM_MEDIA_TYPE*)alloca(cbMT);
		// Get the media type structure.
		hr = spIWMVideoMediaProps->GetMediaType(pType, &cbMT);
		if (FAILED(hr))
			return hr;
        if (pType->formattype != WMFORMAT_VideoInfo)
            return E_FAIL;
                
		WMVIDEOINFOHEADER* pVIH = (WMVIDEOINFOHEADER*)pType->pbFormat;

		// First set pointers to the video structures.
		WMVIDEOINFOHEADER* pVidHdr = (WMVIDEOINFOHEADER*)pType->pbFormat;
		pVidHdr->dwBitRate = dwBitRate;
		pVidHdr->rcSource.right = m_lFrameWidth;
		pVidHdr->rcSource.bottom = m_lFrameHeight;
		pVidHdr->rcTarget.right = m_lFrameWidth;
		pVidHdr->rcTarget.bottom = m_lFrameHeight;

		BITMAPINFOHEADER* pbmi  = &(pVidHdr->bmiHeader);
		pbmi->biWidth  = m_lFrameWidth;
		pbmi->biHeight = m_lFrameHeight;
    
		// Stride = (width * bytes/pixel), rounded to the next DWORD boundary.
		LONG lStride = (m_lFrameWidth * (pbmi->biBitCount / 8) + 3) & ~3;

		// Image size = stride * height. 
		pbmi->biSizeImage = m_lFrameHeight * lStride;

		// Apply the adjusted type to the video input. 
		hr = spIWMVideoMediaProps->SetMediaType(pType);
		if (FAILED(hr))
			return hr;

		spIWMVideoMediaProps->SetQuality(100);
		return hr;
	}
};

class CMainFrame : public CFrameWindowImpl<CMainFrame>, public CUpdateUI<CMainFrame>
{
	HToupCam	m_Htoupcam;
	CMainView	m_view;
	ToupcamInst m_ti[TOUPCAM_MAX];
	unsigned	m_nIndex;
	BOOL		m_bPaused, m_bSnaping;
	unsigned	m_nFrameCount;
	DWORD		m_dwStartTick, m_dwLastTick;

	wchar_t				m_szFilePath[MAX_PATH];

	CWmvRecord*			m_pWmvRecord;
	BYTE*				m_pData;
	BITMAPINFOHEADER	m_header;
public:
	DECLARE_FRAME_WND_CLASS(NULL, IDR_MAIN)

	BEGIN_MSG_MAP_EX(CMainFrame)
		MSG_WM_CREATE(OnCreate)
		MESSAGE_HANDLER(WM_DESTROY, OnWmDestroy)
		MESSAGE_HANDLER(MSG_CAMEVENT, OnMsgCamEvent)
		COMMAND_RANGE_HANDLER_EX(ID_DEVICE_DEVICE0, ID_DEVICE_DEVICEF, OnOpenDevice)
		COMMAND_RANGE_HANDLER_EX(ID_PREVIEW_RESOLUTION0, ID_PREVIEW_RESOLUTION4, OnPreviewResolution)
		COMMAND_RANGE_HANDLER_EX(ID_SNAP_RESOLUTION0, ID_SNAP_RESOLUTION4, OnSnapResolution)
		COMMAND_ID_HANDLER_EX(ID_CONFIG_WHITEBALANCE, OnWhiteBalance)
		COMMAND_ID_HANDLER_EX(ID_CONFIG_AUTOEXPOSURE, OnAutoExposure)
		COMMAND_ID_HANDLER_EX(ID_CONFIG_VERTICALFLIP, OnVerticalFlip)
		COMMAND_ID_HANDLER_EX(ID_CONFIG_HORIZONTALFLIP, OnHorizontalFlip)
		COMMAND_ID_HANDLER_EX(ID_ACTION_PAUSE, OnPause)
		COMMAND_ID_HANDLER_EX(ID_CONFIG_EXPOSURETIME, OnExposureTime)
		COMMAND_ID_HANDLER_EX(ID_ACTION_STARTRECORD, OnStartRecord)
		COMMAND_ID_HANDLER_EX(ID_ACTION_STOPRECORD, OnStopRecord)
		CHAIN_MSG_MAP(CUpdateUI<CMainFrame>)
		CHAIN_MSG_MAP(CFrameWindowImpl<CMainFrame>)
	END_MSG_MAP()

	BEGIN_UPDATE_UI_MAP(CMainFrame)
		UPDATE_ELEMENT(ID_CONFIG_WHITEBALANCE, UPDUI_MENUPOPUP)
		UPDATE_ELEMENT(ID_CONFIG_AUTOEXPOSURE, UPDUI_MENUPOPUP)
		UPDATE_ELEMENT(ID_CONFIG_VERTICALFLIP, UPDUI_MENUPOPUP)
		UPDATE_ELEMENT(ID_CONFIG_HORIZONTALFLIP, UPDUI_MENUPOPUP)
		UPDATE_ELEMENT(ID_ACTION_STARTRECORD, UPDUI_MENUPOPUP)
		UPDATE_ELEMENT(ID_ACTION_STOPRECORD, UPDUI_MENUPOPUP)
		UPDATE_ELEMENT(ID_ACTION_PAUSE, UPDUI_MENUPOPUP)
		UPDATE_ELEMENT(ID_CONFIG_EXPOSURETIME, UPDUI_MENUPOPUP)
		UPDATE_ELEMENT(ID_PREVIEW_RESOLUTION0, UPDUI_MENUPOPUP)
		UPDATE_ELEMENT(ID_PREVIEW_RESOLUTION1, UPDUI_MENUPOPUP)
		UPDATE_ELEMENT(ID_PREVIEW_RESOLUTION2, UPDUI_MENUPOPUP)
		UPDATE_ELEMENT(ID_PREVIEW_RESOLUTION3, UPDUI_MENUPOPUP)
		UPDATE_ELEMENT(ID_PREVIEW_RESOLUTION4, UPDUI_MENUPOPUP)
		UPDATE_ELEMENT(ID_SNAP_RESOLUTION0, UPDUI_MENUPOPUP)
		UPDATE_ELEMENT(ID_SNAP_RESOLUTION1, UPDUI_MENUPOPUP)
		UPDATE_ELEMENT(ID_SNAP_RESOLUTION2, UPDUI_MENUPOPUP)
		UPDATE_ELEMENT(ID_SNAP_RESOLUTION3, UPDUI_MENUPOPUP)
		UPDATE_ELEMENT(ID_SNAP_RESOLUTION4, UPDUI_MENUPOPUP)
	END_UPDATE_UI_MAP()

	LRESULT OnMsgCamEvent(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
	{
		switch (wParam)
		{
		case TOUPCAM_EVENT_ERROR:
			OnEventUsbError();
			break;
		case TOUPCAM_EVENT_EXPOSURE:
			OnEventExpo();
			break;
		case TOUPCAM_EVENT_TEMPTINT:
			OnEventTemptint();
			break;
		case TOUPCAM_EVENT_IMAGE:
			OnEventImage();
			break;
		case TOUPCAM_EVENT_STILLIMAGE:
			OnEventSnap();
			break;
		}
		return 0;
	}

	CMainFrame()
	: m_Htoupcam(NULL), m_nIndex(0), m_bPaused(FALSE), m_bSnaping(FALSE), m_nFrameCount(0), m_dwStartTick(0), m_dwLastTick(0), m_pWmvRecord(NULL), m_pData(NULL), m_view(this)
	{
		memset(m_ti, 0, sizeof(m_ti));
		memset(m_szFilePath, 0, sizeof(m_szFilePath));
		memset(&m_header, 0, sizeof(m_header));

		m_header.biSize = sizeof(BITMAPINFOHEADER);
		m_header.biPlanes = 1;
		m_header.biBitCount = 24;
	}

	int OnCreate(LPCREATESTRUCT /*lpCreateStruct*/)
	{
		CMenuHandle menu = GetMenu();
		CMenuHandle submenu = menu.GetSubMenu(0);
		while (submenu.GetMenuItemCount() > 0)
			submenu.RemoveMenu(submenu.GetMenuItemCount() - 1, MF_BYPOSITION);

		unsigned cnt = Toupcam_Enum(m_ti);
		if (0 == cnt)
			submenu.AppendMenu(MF_GRAYED | MF_STRING, ID_DEVICE_DEVICE0, L"No Device");
		else
		{
			for (unsigned i = 0; i < cnt; ++i)
				submenu.AppendMenu(MF_STRING, ID_DEVICE_DEVICE0 + i, m_ti[i].displayname);
		}

		CreateSimpleStatusBar();
		{
			int iWidth[] = { 100, 400, 600, -1 };
			CStatusBarCtrl statusbar(m_hWndStatusBar);
			statusbar.SetParts(_countof(iWidth), iWidth);
		}

		m_hWndClient = m_view.Create(m_hWnd, rcDefault, NULL, WS_CHILD | WS_VISIBLE | WS_CLIPSIBLINGS | WS_CLIPCHILDREN, WS_EX_CLIENTEDGE);
		
		OnDeviceChanged();
		return 0;
	}

	void OnWhiteBalance(UINT uNotifyCode, int nID, HWND wndCtl)
	{
		if (m_Htoupcam)
			Toupcam_put_TempTintInit(m_Htoupcam, NULL, NULL);
	}

	void OnAutoExposure(UINT uNotifyCode, int nID, HWND wndCtl)
	{
		if (m_Htoupcam)
		{
			BOOL bAutoExposure = FALSE;
			if (SUCCEEDED(Toupcam_get_AutoExpoEnable(m_Htoupcam, &bAutoExposure)))
			{
				bAutoExposure = !bAutoExposure;
				Toupcam_put_AutoExpoEnable(m_Htoupcam, bAutoExposure);
				UISetCheck(ID_CONFIG_AUTOEXPOSURE, bAutoExposure ? 1 : 0);
				UIEnable(ID_CONFIG_EXPOSURETIME, !bAutoExposure);
			}
		}
	}

	void OnVerticalFlip(UINT uNotifyCode, int nID, HWND wndCtl)
	{
		if (m_Htoupcam)
		{
			BOOL b = FALSE;
			if (SUCCEEDED(Toupcam_get_VFlip(m_Htoupcam, &b)))
			{
				b = !b;
				Toupcam_put_VFlip(m_Htoupcam, b);
				UISetCheck(ID_CONFIG_VERTICALFLIP, b ? 1 : 0);
			}
		}
	}

	void OnHorizontalFlip(UINT uNotifyCode, int nID, HWND wndCtl)
	{
		if (m_Htoupcam)
		{
			BOOL b = FALSE;
			if (SUCCEEDED(Toupcam_get_HFlip(m_Htoupcam, &b)))
			{
				b = !b;
				Toupcam_put_HFlip(m_Htoupcam, b);
				UISetCheck(ID_CONFIG_HORIZONTALFLIP, b ? 1 : 0);
			}
		}
	}

	void OnPause(UINT uNotifyCode, int nID, HWND wndCtl)
	{
		if (m_Htoupcam)
		{
			m_bPaused = !m_bPaused;
			Toupcam_Pause(m_Htoupcam, m_bPaused);
			
			UISetCheck(ID_ACTION_PAUSE, m_bPaused ? 1 : 0);
			UIEnable(ID_ACTION_STARTRECORD, !m_bPaused);
		}
	}

	void OnExposureTime(UINT uNotifyCode, int nID, HWND wndCtl)
	{
		if (m_Htoupcam)
		{
			CExposureTimeDlg dlg(&m_Htoupcam);
			if (IDOK == dlg.DoModal())
				UpdateExposureTimeText();
		}
	}

	void OnPreviewResolution(UINT uNotifyCode, int nID, HWND wndCtl)
	{
		if (NULL == m_Htoupcam)
			return;

		unsigned eSize = 0;
		if (SUCCEEDED(Toupcam_get_eSize(m_Htoupcam, &eSize)))
		{
			if (eSize != nID - ID_PREVIEW_RESOLUTION0)
			{
				if (SUCCEEDED(Toupcam_Stop(m_Htoupcam)))
				{
					OnStopRecord(0, 0, NULL);

					m_bPaused = m_bSnaping = FALSE;
					UISetCheck(ID_ACTION_PAUSE, FALSE);
					m_nFrameCount = 0;
					m_dwStartTick = m_dwLastTick = 0;

					Toupcam_put_eSize(m_Htoupcam, nID - ID_PREVIEW_RESOLUTION0);
					for (unsigned i = 0; i < m_ti[m_nIndex].model->preview; ++i)
						UISetCheck(ID_PREVIEW_RESOLUTION0 + i, (nID - ID_PREVIEW_RESOLUTION0 == i) ? 1 : 0);
					UpdateSnapMenu();
					if (SUCCEEDED(Toupcam_get_Size(m_Htoupcam, (int*)&m_header.biWidth, (int*)&m_header.biHeight)))
					{
						UpdateResolutionText();
						UpdateFrameText();
						UpdateExposureTimeText();

						m_header.biSizeImage = TDIBWIDTHBYTES(m_header.biWidth * m_header.biBitCount) * m_header.biHeight;
						if (m_pData)
						{
							free(m_pData);
							m_pData = NULL;
						}
						m_pData = (BYTE*)malloc(m_header.biSizeImage);
						if (SUCCEEDED(Toupcam_StartPullModeWithWndMsg(m_Htoupcam, m_hWnd, MSG_CAMEVENT)))
						{
							UIEnable(ID_ACTION_PAUSE, TRUE);
							UIEnable(ID_ACTION_STARTRECORD, TRUE);
						}
					}
				}
			}
		}
	}

	void OnSnapResolution(UINT uNotifyCode, int nID, HWND wndCtl)
	{
		if (NULL == m_Htoupcam)
			return;

		CFileDialog dlg(FALSE, L"jpg");
		if (IDOK == dlg.DoModal())
		{
			wcscpy(m_szFilePath, dlg.m_szFileName);
			if (SUCCEEDED(Toupcam_Snap(m_Htoupcam, nID - ID_SNAP_RESOLUTION0)))
			{
				m_bSnaping = TRUE;
				UpdateSnapMenu();
			}
		}
	}

	void OnOpenDevice(UINT uNotifyCode, int nID, HWND wndCtl)
	{
		CloseDevice();

		m_bPaused = m_bSnaping = FALSE;
		UISetCheck(ID_ACTION_PAUSE, FALSE);
		m_nFrameCount = 0;
		m_dwStartTick = m_dwLastTick = 0;
		m_nIndex = nID - ID_DEVICE_DEVICE0;
		m_Htoupcam = Toupcam_Open(m_ti[m_nIndex].id);
		if (m_Htoupcam)
		{
			OnDeviceChanged();
			UpdateFrameText();

			if (SUCCEEDED(Toupcam_get_Size(m_Htoupcam, (int*)&m_header.biWidth, (int*)&m_header.biHeight)))
			{
				m_header.biSizeImage = TDIBWIDTHBYTES(m_header.biWidth * m_header.biBitCount) * m_header.biHeight;
				m_pData = (BYTE*)malloc(m_header.biSizeImage);
				unsigned eSize = 0;
				if (SUCCEEDED(Toupcam_get_eSize(m_Htoupcam, &eSize)))
				{
					for (unsigned i = 0; i < m_ti[m_nIndex].model->preview; ++i)
						UISetCheck(ID_PREVIEW_RESOLUTION0 + i, (eSize == i) ? 1 : 0);
				}
				if (SUCCEEDED(Toupcam_StartPullModeWithWndMsg(m_Htoupcam, m_hWnd, MSG_CAMEVENT)))
				{
					UIEnable(ID_ACTION_PAUSE, TRUE);
					UIEnable(ID_ACTION_STARTRECORD, TRUE);
				}
			}
		}
	}

	void OnStartRecord(UINT uNotifyCode, int nID, HWND wndCtl)
	{
		CFileDialog dlg(FALSE, L"wmv");
		if (IDOK == dlg.DoModal())
		{
			StopRecord();

			DWORD dwBitrate = 4 * 1024 * 1024; /* bitrate, you can change this setting */
			CWmvRecord* pWmvRecord = new CWmvRecord(m_header.biWidth, m_header.biHeight);
			if (pWmvRecord->StartRecord(dlg.m_szFileName, dwBitrate))
			{
				m_pWmvRecord = pWmvRecord;
				UIEnable(ID_ACTION_STARTRECORD, FALSE);
				UIEnable(ID_ACTION_STOPRECORD, TRUE);
			}
			else
			{
				delete pWmvRecord;
			}
		}
	}

	void OnStopRecord(UINT uNotifyCode, int nID, HWND wndCtl)
	{
		StopRecord();

		UIEnable(ID_ACTION_STARTRECORD, m_Htoupcam ? TRUE : FALSE);
		UIEnable(ID_ACTION_STOPRECORD, FALSE);
	}

	void OnEventImage()
	{
		HRESULT hr = Toupcam_PullImage(m_Htoupcam, m_pData, 24, NULL, NULL);
		if (FAILED(hr))
			return;

		++m_nFrameCount;
		if (0 == m_dwStartTick)
		{
			m_dwLastTick = m_dwStartTick = GetTickCount();
		}
		else
		{
			m_dwLastTick = GetTickCount();
		}
		m_view.Invalidate();

		UpdateFrameText();
		if (m_pWmvRecord)
			m_pWmvRecord->WriteSample(m_pData);
	}

	void OnEventSnap()
	{
		BITMAPINFOHEADER header = { 0 };
		header.biSize = sizeof(header);
		header.biPlanes = 1;
		header.biBitCount = 24;
		HRESULT hr = Toupcam_PullStillImage(m_Htoupcam, NULL, 24, (unsigned*)&header.biWidth, (unsigned*)&header.biHeight); //first, peek the width and height
		if (SUCCEEDED(hr))
		{
			header.biSizeImage = TDIBWIDTHBYTES(header.biWidth * header.biBitCount) * header.biHeight;
			void* pSnapData = malloc(header.biSizeImage);
			if (pSnapData)
			{
				hr = Toupcam_PullStillImage(m_Htoupcam, pSnapData, 24, NULL, NULL);
				if (SUCCEEDED(hr))
				{
					if (PathMatchSpec(m_szFilePath, L"*.bmp"))
						SaveBmp(m_szFilePath, pSnapData, &header);
					else
						SaveByWIC(m_szFilePath, pSnapData, &header);
				}

				free(pSnapData);
			}
		}

		m_bSnaping = FALSE;
		UpdateSnapMenu();
	}

	LRESULT OnWmDestroy(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
	{
		CloseDevice();

		CFrameWindowImpl<CMainFrame>::OnDestroy(uMsg, wParam, lParam, bHandled);
		return 0;
	}

	void OnEventUsbError()
	{
		CloseDevice();
		AtlMessageBox(m_hWnd, _T("Error"));
	}

	void OnEventTemptint()
	{
		CStatusBarCtrl statusbar(m_hWndStatusBar);
		wchar_t res[128];
		int nTemp = 0, nTint = 0;
		Toupcam_get_TempTint(m_Htoupcam, &nTemp, &nTint);
		swprintf(res, L"Temp = %d, Tint = %d", nTemp, nTint);
		statusbar.SetText(2, res);
	}

	void OnEventExpo()
	{
		CStatusBarCtrl statusbar(m_hWndStatusBar);
		wchar_t res[128];
		unsigned nTime = 0;
		unsigned short AGain = 0;
		if (SUCCEEDED(Toupcam_get_ExpoTime(m_Htoupcam, &nTime)) && SUCCEEDED(Toupcam_get_ExpoAGain(m_Htoupcam, &AGain)))
		{
			swprintf(res, L"ExposureTime = %u, AGain = %hu", nTime, AGain);
			statusbar.SetText(1, res);
		}
	}

	BOOL GetData(BITMAPINFOHEADER** pHeader, BYTE** pData)
	{
		if (m_pData)
		{
			*pData = m_pData;
			*pHeader = &m_header;
			return TRUE;
		}

		return FALSE;
	}

private:
	void CloseDevice()
	{
		OnStopRecord(0, 0, NULL);

		if (m_Htoupcam)
		{
			Toupcam_Close(m_Htoupcam);
			m_Htoupcam = NULL;

			if (m_pData)
			{
				free(m_pData);
				m_pData = NULL;
			}
		}
		OnDeviceChanged();
	}

	void OnDeviceChanged()
	{
		CMenuHandle menu = GetMenu();
		CMenuHandle submenu = menu.GetSubMenu(1);
		CMenuHandle previewsubmenu = submenu.GetSubMenu(0);
		CMenuHandle snapsubmenu = submenu.GetSubMenu(1);
		while (previewsubmenu.GetMenuItemCount() > 0)
			previewsubmenu.RemoveMenu(previewsubmenu.GetMenuItemCount() - 1, MF_BYPOSITION);
		while (snapsubmenu.GetMenuItemCount() > 0)
			snapsubmenu.RemoveMenu(snapsubmenu.GetMenuItemCount() - 1, MF_BYPOSITION);

		CStatusBarCtrl statusbar(m_hWndStatusBar);

		if (NULL == m_Htoupcam)
		{
			previewsubmenu.AppendMenu(MF_STRING | MF_GRAYED, ID_PREVIEW_RESOLUTION0, L"Empty");
			snapsubmenu.AppendMenu(MF_STRING | MF_GRAYED, ID_SNAP_RESOLUTION0, L"Empty");

			statusbar.SetText(0, L"");
			statusbar.SetText(1, L"");
			statusbar.SetText(2, L"");
			statusbar.SetText(3, L"");

			UISetCheck(ID_CONFIG_AUTOEXPOSURE, 0);
			UIEnable(ID_CONFIG_EXPOSURETIME, FALSE);
			UIEnable(ID_PREVIEW_RESOLUTION0, FALSE);
			UIEnable(ID_SNAP_RESOLUTION0, FALSE);
		}
		else
		{
			unsigned eSize = 0;
			Toupcam_get_eSize(m_Htoupcam, &eSize);

			wchar_t res[128];
			for (unsigned i = 0; i < m_ti[m_nIndex].model->preview; ++i)
			{
				swprintf(res, L"%u * %u", m_ti[m_nIndex].model->res[i].width, m_ti[m_nIndex].model->res[i].height);
				previewsubmenu.AppendMenu(MF_STRING, ID_PREVIEW_RESOLUTION0 + i, res);
				snapsubmenu.AppendMenu(MF_STRING, ID_SNAP_RESOLUTION0 + i, res);

				UIEnable(ID_PREVIEW_RESOLUTION0 + i, TRUE);
			}
			UpdateSnapMenu();

			UpdateResolutionText();
			UpdateExposureTimeText();

			int nTemp = 0, nTint = 0;
			if (SUCCEEDED(Toupcam_get_TempTint(m_Htoupcam, &nTemp, &nTint)))
			{
				swprintf(res, L"Temp = %d, Tint = %d", nTemp, nTint);
				statusbar.SetText(2, res);
			}

			BOOL bAutoExposure = TRUE;
			if (SUCCEEDED(Toupcam_get_AutoExpoEnable(m_Htoupcam, &bAutoExposure)))
			{
				UISetCheck(ID_CONFIG_AUTOEXPOSURE, bAutoExposure ? 1 : 0);
				UIEnable(ID_CONFIG_EXPOSURETIME, !bAutoExposure);
			}
		}

		UIEnable(ID_ACTION_PAUSE, FALSE);
		UIEnable(ID_ACTION_STARTRECORD, FALSE);
		UIEnable(ID_ACTION_STOPRECORD, FALSE);
		UIEnable(ID_CONFIG_AUTOEXPOSURE, m_Htoupcam ? TRUE : FALSE);
		UIEnable(ID_CONFIG_HORIZONTALFLIP, m_Htoupcam ? TRUE : FALSE);
		UIEnable(ID_CONFIG_VERTICALFLIP, m_Htoupcam ? TRUE : FALSE);
		UIEnable(ID_CONFIG_WHITEBALANCE, m_Htoupcam ? TRUE : FALSE);
		UISetCheck(ID_ACTION_PAUSE, 0);
		UISetCheck(ID_CONFIG_HORIZONTALFLIP, 0);
		UISetCheck(ID_CONFIG_VERTICALFLIP, 0);
	}

	void UpdateSnapMenu()
	{
		if (m_bSnaping)
		{
			for (unsigned i = 0; i < m_ti[m_nIndex].model->preview; ++i)
				UIEnable(ID_SNAP_RESOLUTION0 + i, FALSE);
			return;
		}

		unsigned eSize = 0;
		if (SUCCEEDED(Toupcam_get_eSize(m_Htoupcam, &eSize)))
		{
			for (unsigned i = 0; i < m_ti[m_nIndex].model->preview; ++i)
			{
				if (m_ti[m_nIndex].model->still == m_ti[m_nIndex].model->preview) /* still capture full supported */
				{
					UIEnable(ID_SNAP_RESOLUTION0 + i, TRUE);
				}
				else if (0 == m_ti[m_nIndex].model->still) /* still capture not supported */
				{
					UIEnable(ID_SNAP_RESOLUTION0 + i, (eSize == i) ? TRUE : FALSE);
				}
				else if (m_ti[m_nIndex].model->still < m_ti[m_nIndex].model->preview)
				{
					if ((eSize == i) || (i < m_ti[m_nIndex].model->still))
						UIEnable(ID_SNAP_RESOLUTION0 + i, TRUE);
					else
						UIEnable(ID_SNAP_RESOLUTION0 + i, FALSE);
				}
			}
		}
	}

	void UpdateResolutionText()
	{
		CStatusBarCtrl statusbar(m_hWndStatusBar);
		wchar_t res[128];
		int nWidth = 0, nHeight = 0;
		if (SUCCEEDED(Toupcam_get_Size(m_Htoupcam, &nWidth, &nHeight)))
		{
			swprintf(res, L"%u * %u", nWidth, nHeight);
			statusbar.SetText(0, res);
		}
	}

	void UpdateFrameText()
	{
		CStatusBarCtrl statusbar(m_hWndStatusBar);
		wchar_t str[256];
		if (m_dwLastTick != m_dwStartTick)
			swprintf(str, L"%u, %.2f", m_nFrameCount, m_nFrameCount / ((m_dwLastTick - m_dwStartTick) / 1000.0));
		else
			swprintf(str, L"%u", m_nFrameCount);
		statusbar.SetText(3, str);
	}

	void UpdateExposureTimeText()
	{
		CStatusBarCtrl statusbar(m_hWndStatusBar);
		wchar_t res[128];
		unsigned nTime = 0;
		unsigned short AGain = 0;
		if (SUCCEEDED(Toupcam_get_ExpoTime(m_Htoupcam, &nTime)) && SUCCEEDED(Toupcam_get_ExpoAGain(m_Htoupcam, &AGain)))
		{
			swprintf(res, L"ExposureTime = %u, AGain = %hu", nTime, AGain);
			statusbar.SetText(1, res);
		}
	}

	/* this is called in the UI thread */
	void StopRecord()
	{
		if (m_pWmvRecord)
		{
			m_pWmvRecord->StopRecord();

			delete m_pWmvRecord;
			m_pWmvRecord = NULL;
		}
	}
};

LRESULT CMainView::OnWmPaint(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	CPaintDC dc(m_hWnd);

	RECT rc;
	GetClientRect(&rc);
	BITMAPINFOHEADER* pHeader = NULL;
	BYTE* pData = NULL;
	if (m_pMainFrame->GetData(&pHeader, &pData))
	{
		int m = dc.SetStretchBltMode(COLORONCOLOR);
		StretchDIBits(dc, rc.left, rc.top, rc.right - rc.left, rc.bottom - rc.top, 0, 0, pHeader->biWidth, pHeader->biHeight, pData, (BITMAPINFO*)pHeader, DIB_RGB_COLORS, SRCCOPY);
		dc.SetStretchBltMode(m);
	}
	else
	{
		dc.FillRect(&rc, (HBRUSH)WHITE_BRUSH);
	}

	return 0;
}

static int Run(int nCmdShow = SW_SHOWDEFAULT)
{
	CMessageLoop theLoop;
	_Module.AddMessageLoop(&theLoop);

	CMainFrame frmMain;

	if (frmMain.CreateEx() == NULL)
	{
		ATLTRACE(_T("Main window creation failed!\n"));
		return 0;
	}

	frmMain.ShowWindow(nCmdShow);

	int nRet = theLoop.Run();

	_Module.RemoveMessageLoop();
	return nRet;
}

int WINAPI _tWinMain(HINSTANCE hInstance, HINSTANCE /*hPrevInstance*/, LPTSTR /*pCmdLine*/, int nCmdShow)
{
	INITCOMMONCONTROLSEX iccx;
	iccx.dwSize = sizeof(iccx);
	iccx.dwICC = ICC_COOL_CLASSES | ICC_BAR_CLASSES;
	InitCommonControlsEx(&iccx);

	OleInitialize(NULL);

	_Module.Init(NULL, hInstance);

	int nRet = Run(nCmdShow);

	_Module.Term();
	return nRet;
}